class EUpload {
    static defaultConfig = {
        url: '',
        chunk: 2, // 2MB
        size: 0, // MB
        filenum: 0, // MB
        autoup: true,
        exp: 'file_no',
        retryTimes: 3,
    };
    constructor(config = {}) {
        this.config = this.mergeConfig(config);
        this.state = {
            files: new Map(),
            controller: new AbortController(),
            uploadQueue: []
        };
        $(this.config.monitor).click(() => this.createInput());
        $(this.config.monitor).click(() => this.createInput());
    }
    // 配置合并
    mergeConfig(userConfig) {
        const domConfig = $(userConfig.monitor).data();
        return {
            ...EUpload.defaultConfig,
            ...userConfig,
            ...domConfig
        };
    }
    // 初始化界面
    UpUI(selector) {
        const container = $(selector).parent();
        // 绑定事件
        $('#uploadBtn').click(() => this.createInput());
        $('#pauseBtn').click(() => this.pauseAll());
        $('#resumeBtn').click(() => this.resumeAll());

        // 创建预览容器
        container.append('<div id="previewContainer"></div>');
    }
    // 创建文件输入
    createInput() {
        const input = document.createElement('input');
        let iptconf = {
            type: 'file',
            style: `display: none;`,
            accept: '',
            multiple: true
        };
        if(this.config.filenum > '0') {
            iptconf.multiple = this.config.filenum
        }
        Object.assign(input, iptconf);

        if(typeof this.initUI === 'function') {
            this.initUI(this.config)
        } else {
            this.UpUI(this.config.monitor);
        }
        // console.log(typeof this.initUI);

        input.addEventListener('change', async e => {
            try {
                const files = await this.processFiles(e.target.files);
                if (this.config.autoup){
                    this.uploadAll(files);
                }
            } catch (error) {
                this.showError(error.message);
            } finally {
                input.remove();
            }
        });
        document.body.appendChild(input).click();
    }
    // 文件预处理
    async processFiles(files) {
        const validFiles = [];
        if(this.config.filenum > '0' && files.length > this.config.filenum) {
            this.showError(`文件数量超过限制,最多${this.config.filenum}个`);
            return false;
        }
        for (const file of files) {
            if (!this.validateFile(file)) continue;

            const fileKey = await this.generateFileKey(file);
            if (this.state.files.has(fileKey)) continue;

            const fileState = {
                file,
                chunks: this.createChunks(file),
                index: 0,
                status: 'pending',
                md5: fileKey,
                progress: 0,
                controller: new AbortController(), // 每个文件独立的中止控制器
                isUserPaused: false                 // 用户手动暂停标记
            };
            this.state.files.set(fileKey, fileState);
            if(typeof this.process == 'function'){
                fileState.icon = URL.createObjectURL(fileState.file);
                let p = this.process(fileState);
                if(p && p.paused){
                    this.AddPause(p.paused,fileState);
                }
                if(p && p.del){
                    this.AddDelete(p.del,fileState);
                }
            } else {
                this.createPreview(fileState);
            }
            validFiles.push(fileState);
        }
        return validFiles;
    }
    // 文件验证
    validateFile(file) {
        // 文件大小验证
        if (this.config.size > '0' && file.size > this.config.size * 1024 * 1024) {
            this.showError(`文件大小超过限制: ${file.name}`);
            return false;
        }
        let fileExp = file.type.substr(file.type.lastIndexOf("/") + 1).toLowerCase(),
            fileTypeString = this.getSupportedExtensions(this.config.exp);

        // 如果 fileTypeString 不是字符串，则视为没有设置文件格式限制
        if (typeof fileTypeString !== 'string') {
            return true;
        }
        // 如果 fileTypeString 不是字符串，则视为没有设置文件格式限制
        if (this.matchesFileType(fileExp, fileTypeString)) {
            return true;
        }
        this.showError(`文件格式不支持`);
        return false;
    }
    /**
     * 生成文件唯一标识（修复RangeError问题）
     * 使用增量哈希计算，避免大数组问题
     *
     * @param {File} file - 待处理文件
     * @returns {Promise&lt;string&gt;} - 文件唯一标识
     */
    async generateFileKey(file) {
        return new Promise(resolve => {
            const spark = new SparkMD5.ArrayBuffer();
            const reader = new FileReader();

            reader.onload = e => {
                spark.append(e.target.result);
                resolve(spark.end());
            };

            reader.readAsArrayBuffer(file); // 取文件头计算
        });
    }
    // 创建分片
    createChunks(file) {
        const chunks = [];
        let offset = 0;

        while (offset < file.size) {
            const end = Math.min(offset + this.config.chunk * 1024 * 1024, file.size);
            chunks.push(file.slice(offset, end));
            offset = end;
        }

        return chunks;
    }
    // 创建预览
    createPreview(fileState) {
        const preview = $(`
            <div class="eupload-preview" data-key="${fileState.md5}">
                <img class="preview-thumbnail">
                <div class="file-meta">
                    <div class="file-name">${fileState.file.name}</div>
                    <div class="file-size">${this.formatSize(fileState.file.size)}</div>
                </div>
                <div class="progress-container">
                    <div class="progress-bar"></div>
                </div>
                <div class="status-text"></div>
                <div class="pause-btn">暂停</div>
            </div>
        `);
        // 绑定暂停/恢复点击事件
        preview.find('.pause-btn').click(() => {
            if (fileState.status === 'uploading') {
                this.pauseFile(fileState);
            } else if (fileState.status === 'paused') {
                this.resumeFile(fileState);
            }
        });
        // 生成缩略图
        if (fileState.file.type.startsWith('image/')) {
            const url = URL.createObjectURL(fileState.file);
            preview.find('img').attr('src', url).on('load', () => URL.revokeObjectURL(url));
        }
        $('#previewContainer').append(preview);
    }
    //  注册暂停事件
    AddPause(pausename,fileState){
        let btn = $(`.${pausename.btn}`);
        btn.click(() => {
            if (fileState.status === 'uploading') {
                if(pausename.uploading_title){
                    btn.text(pausename.uploading_title)
                }
                this.pauseFile(fileState);
            } else if (fileState.status === 'paused') {
                if(pausename.paused_title){
                    btn.text(pausename.paused_title)
                }
                this.resumeFile(fileState);
            }
        });
    }
    //  注册删除事件
    AddDelete(pausename,fileState){
        $(`.${pausename}`).click(() => {
            if (confirm('确认删除此文件？')) {
                this.deleteFile(fileState.md5);
                if(typeof this.Del == 'function'){
                    this.Del(pausename,fileState);
                }
            }
        })
    }
    // 上传管理
    async uploadAll(files) {
        for (const fileState of files) {
            if (typeof this.before === 'function') {
                const proceed = await this.before(fileState);
                if (!proceed) {
                    this.updateStatus(fileState, 'canceled');
                    continue;
                }
            }

            this.state.uploadQueue.push(fileState);
            this.processQueue();
        }
    }
    // 上传队列处理
    async processQueue() {
        while (this.state.uploadQueue.length > 0) {
            const fileState = this.state.uploadQueue.shift();
            if (fileState.status === 'paused'){
                continue;
            }
            try {
                let res = await this.uploadFile(fileState);
                if(typeof this.success === 'function') {
                    this.success(res, fileState);
                } else if (fileState.index === fileState.chunks.length) {
                    this.updateStatus(fileState, 'completed');
                }
            } catch (error) {
                this.updateStatus(fileState, 'error');
            }
        }
    }
    // 单个文件上传
    // uploadFile 方法修改
    async uploadFile(fileState) {
        let res;
        try {
            let retryCount = 0;
            const maxRetries = this.config.retryTimes || 3;
            fileState.status = 'uploading';
            this.updateControlButton(fileState);
            for (; fileState.index < fileState.chunks.length; fileState.index++) {
                if (fileState.status === 'paused' || this.state.isAllPaused){
                    break;
                }
                const chunk = fileState.chunks[fileState.index];
                // 重试逻辑
                while (retryCount < maxRetries) {
                    try {
                        res = await this.uploadChunk(chunk, fileState,true);
                        // 检查业务状态码（兼容数字和字符串）
                        if (res.code.toString() !== '1') {
                            throw new Error(`业务错误: ${res.message || '未知原因'}`);
                        }
                        break; // 成功则退出重试循环
                    } catch (error) {
                        retryCount++;
                        if (retryCount >= maxRetries) {
                            throw error; // 重试耗尽，抛出错误
                        }
                        console.warn(`分块 ${fileState.index} 第 ${retryCount} 次重试...`);
                    }
                }
                // 更新进度（基于分块大小）
                const uploadedSize = fileState.chunks.slice(0, fileState.index + 1).reduce((sum, c) => sum + c.size, 0),
                    progress = (uploadedSize / fileState.file.size) * 100;
                if(typeof this.done === 'function') {
                    this.done({progress:progress,md5: fileState.md5});
                } else {
                    this.updateProgress(fileState, progress);
                }
            }

            // if (fileState.status !== 'paused') {
            //     console.log(`上传完成一`)
            //     this.updateStatus(fileState, 'completed');
            // }
        } catch (error) {
            if (error.name !== 'AbortError') {
                fileState.status = 'paused';
                this.updateStatus(fileState, 'paused', error.message);
            } else {
                this.updateStatus(fileState, 'error', '请求被中止');
            }
            throw error;
        }
        return res;
    }
    // 分片上传
    async uploadChunk(chunk, fileState,check = false) {
        const form = new FormData();
        if(!check) {
            form.append('file', chunk);
        } else {
            form.append('checkname', fileState.md5);
        }
        form.append('name', fileState.file.name);
        form.append('type', fileState.file.type);
        form.append('index', fileState.index);
        form.append('total', fileState.chunks.length);
        form.append('md5', fileState.md5);

        try {
            const response = await fetch(this.config.url, {
                method: 'POST',
                body: form,
                signal: this.state.controller.signal
            });
            if (!response.ok){
                const errorBody = await response.json().catch(() => ({}));
                throw new Error(errorBody.message || `HTTP Error: ${response.status}`);
            }
            const res = await response.json();
            // 如果未上传,则继续上传
            if(check && res.code.toString() === '0') {
                return this.uploadChunk(chunk, fileState);
            }
            return res;
        } catch (error) {
            if (error.name === 'AbortError') {
                error.isAbort = true; // 标记中止错误
            }
            throw error;
        }
    }
    // 控制方法
    pauseAll() {
        this.state.files.forEach(state => {
            if (state.status === 'uploading') {
                state.status = 'paused';
                this.updateStatus(state, 'paused');
            }
        });
    }
    resumeAll() {
        this.state.files.forEach(state => {
            if (state.status === 'paused') {
                state.status = 'pending';
                this.state.uploadQueue.push(state);
            }
        });
        this.processQueue();
    }
    // 单文件暂停
    pauseFile(fileState) {
        if (fileState.status === 'uploading') {
            fileState.status = 'paused';
            fileState.isUserPaused = true;
            fileState.controller.abort(); // 中止当前分块上传
            this.updateStatus(fileState, 'paused');
            this.updateControlButton(fileState); // 更新按钮文本
        }
    }
    //  单文件恢复上传
    resumeFile(fileState) {
        if (fileState.status === 'paused' && fileState.isUserPaused) {
            fileState.status = 'pending';
            fileState.isUserPaused = false;
            fileState.controller = new AbortController(); // 重置控制器
            this.state.uploadQueue.push(fileState);
            this.processQueue();
            this.updateControlButton(fileState);
        }
    }
    //  删除文件
    deleteFile(fileKey) {
        const fileState = this.state.files.get(fileKey);
        if (!fileState) return;
        // 中止上传请求
        if (fileState.status === 'uploading') {
            fileState.controller.abort();
        }
        // 从上传队列中移除
        this.state.uploadQueue = this.state.uploadQueue.filter(f => f.md5 !== fileKey);
        // 从文件列表中删除
        this.state.files.delete(fileKey);
        // 移除预览元素
        $(`[data-key="${fileKey}"]`).remove();
        // 触发删除回调
        if (typeof this.Delete === 'function') {
            this.Delete({
                md5: fileKey,
                file: fileState.file,
                status: fileState.status
            });
        }
    }
    //更新按钮状态反馈
    updateControlButton(fileState) {
        const btn = $(`[data-key="${fileState.md5}"] .pause-btn`);
        if (fileState.status === 'uploading') {
            btn.text('暂停');
        } else if (fileState.status === 'paused') {
            btn.text('继续');
        }
    }
    // 状态更新
    updateStatus(fileState, status, message = '') {
        if (!this.state.files.has(fileState.md5)) return;
        fileState.status = status;
        const preview = $(`[data-key="${fileState.md5}"]`);
        preview.find('.status-text').removeClass('error paused aborted').addClass(status === 'error' ? 'error' : '').addClass(status === 'paused' ? 'paused' : '').text(message || this.statusText[status] || '');
    }
    updateProgress(fileState, progress) {
        fileState.progress = progress;
        $(`[data-key="${fileState.md5}"] .progress-bar`).css('width', `${progress}%`);
    }
    // 工具方法
    formatSize(bytes) {
        const units = ['B', 'KB', 'MB', 'GB'];
        let size = bytes;
        let unitIndex = 0;

        while (size >= 1024 && unitIndex < units.length - 1) {
            size /= 1024;
            unitIndex++;
        }

        return `${size.toFixed(1)} ${units[unitIndex]}`;
    }
    showError(message) {
        alert(message);
        const errorDiv = $('<div class="error">').text(message);
        $('#previewContainer').append(errorDiv);
        setTimeout(() => errorDiv.remove(), 3000);
    }
    matchesFileType(fileExp, fileTypeString) {
        let regex = new RegExp(fileTypeString.split(',').join('|'), 'i');
        return regex.test(fileExp);
    }
    // 获取支持的文件类型字符串
    getSupportedExtensions(type) {
        switch (type) {
            case 'file':
                return "pdf|msword|vnd.openxmlformats-officedocument.wordprocessingml.document|vnd.ms-powerpoint|vnd.openxmlformats-officedocument.presentationml.presentation|vnd.ms-excel|vnd.openxmlformats-officedocument.spreadsheetml.sheet|plain";
            case 'video':
                return "avi|mp4|wma|rmvb|rm|flash|3gp|flv|mp3|wav|mpeg";
            case 'image':
                return "jpg|jpeg|png|gif|bmp|webp";
            case 'zip':
                return "tar.gz|zip|x-rar-compressed|x-gzip";
            default:
                return ""; // 如果不支持的类型，默认返回空字符串，这样 matchesFileType 会返回 false
        }
    }
    statusText = {
        pending: '等待中',
        uploading: '上传中',
        paused: '已暂停',
        completed: '已完成',
        error: '上传失败',
        canceled: '已取消',
        retrying: '重试中...'
    };
}

(function(factory){if(typeof exports==="object"){module.exports=factory()}else if(typeof define==="function"&&define.amd){define(factory)}else{var glob;try{glob=window}catch(e){glob=self}glob.SparkMD5=factory()}})(function(undefined){"use strict";var add32=function(a,b){return a+b&4294967295},hex_chr=["0","1","2","3","4","5","6","7","8","9","a","b","c","d","e","f"];function cmn(q,a,b,x,s,t){a=add32(add32(a,q),add32(x,t));return add32(a<<s|a>>>32-s,b)}function md5cycle(x,k){var a=x[0],b=x[1],c=x[2],d=x[3];a+=(b&c|~b&d)+k[0]-680876936|0;a=(a<<7|a>>>25)+b|0;d+=(a&b|~a&c)+k[1]-389564586|0;d=(d<<12|d>>>20)+a|0;c+=(d&a|~d&b)+k[2]+606105819|0;c=(c<<17|c>>>15)+d|0;b+=(c&d|~c&a)+k[3]-1044525330|0;b=(b<<22|b>>>10)+c|0;a+=(b&c|~b&d)+k[4]-176418897|0;a=(a<<7|a>>>25)+b|0;d+=(a&b|~a&c)+k[5]+1200080426|0;d=(d<<12|d>>>20)+a|0;c+=(d&a|~d&b)+k[6]-1473231341|0;c=(c<<17|c>>>15)+d|0;b+=(c&d|~c&a)+k[7]-45705983|0;b=(b<<22|b>>>10)+c|0;a+=(b&c|~b&d)+k[8]+1770035416|0;a=(a<<7|a>>>25)+b|0;d+=(a&b|~a&c)+k[9]-1958414417|0;d=(d<<12|d>>>20)+a|0;c+=(d&a|~d&b)+k[10]-42063|0;c=(c<<17|c>>>15)+d|0;b+=(c&d|~c&a)+k[11]-1990404162|0;b=(b<<22|b>>>10)+c|0;a+=(b&c|~b&d)+k[12]+1804603682|0;a=(a<<7|a>>>25)+b|0;d+=(a&b|~a&c)+k[13]-40341101|0;d=(d<<12|d>>>20)+a|0;c+=(d&a|~d&b)+k[14]-1502002290|0;c=(c<<17|c>>>15)+d|0;b+=(c&d|~c&a)+k[15]+1236535329|0;b=(b<<22|b>>>10)+c|0;a+=(b&d|c&~d)+k[1]-165796510|0;a=(a<<5|a>>>27)+b|0;d+=(a&c|b&~c)+k[6]-1069501632|0;d=(d<<9|d>>>23)+a|0;c+=(d&b|a&~b)+k[11]+643717713|0;c=(c<<14|c>>>18)+d|0;b+=(c&a|d&~a)+k[0]-373897302|0;b=(b<<20|b>>>12)+c|0;a+=(b&d|c&~d)+k[5]-701558691|0;a=(a<<5|a>>>27)+b|0;d+=(a&c|b&~c)+k[10]+38016083|0;d=(d<<9|d>>>23)+a|0;c+=(d&b|a&~b)+k[15]-660478335|0;c=(c<<14|c>>>18)+d|0;b+=(c&a|d&~a)+k[4]-405537848|0;b=(b<<20|b>>>12)+c|0;a+=(b&d|c&~d)+k[9]+568446438|0;a=(a<<5|a>>>27)+b|0;d+=(a&c|b&~c)+k[14]-1019803690|0;d=(d<<9|d>>>23)+a|0;c+=(d&b|a&~b)+k[3]-187363961|0;c=(c<<14|c>>>18)+d|0;b+=(c&a|d&~a)+k[8]+1163531501|0;b=(b<<20|b>>>12)+c|0;a+=(b&d|c&~d)+k[13]-1444681467|0;a=(a<<5|a>>>27)+b|0;d+=(a&c|b&~c)+k[2]-51403784|0;d=(d<<9|d>>>23)+a|0;c+=(d&b|a&~b)+k[7]+1735328473|0;c=(c<<14|c>>>18)+d|0;b+=(c&a|d&~a)+k[12]-1926607734|0;b=(b<<20|b>>>12)+c|0;a+=(b^c^d)+k[5]-378558|0;a=(a<<4|a>>>28)+b|0;d+=(a^b^c)+k[8]-2022574463|0;d=(d<<11|d>>>21)+a|0;c+=(d^a^b)+k[11]+1839030562|0;c=(c<<16|c>>>16)+d|0;b+=(c^d^a)+k[14]-35309556|0;b=(b<<23|b>>>9)+c|0;a+=(b^c^d)+k[1]-1530992060|0;a=(a<<4|a>>>28)+b|0;d+=(a^b^c)+k[4]+1272893353|0;d=(d<<11|d>>>21)+a|0;c+=(d^a^b)+k[7]-155497632|0;c=(c<<16|c>>>16)+d|0;b+=(c^d^a)+k[10]-1094730640|0;b=(b<<23|b>>>9)+c|0;a+=(b^c^d)+k[13]+681279174|0;a=(a<<4|a>>>28)+b|0;d+=(a^b^c)+k[0]-358537222|0;d=(d<<11|d>>>21)+a|0;c+=(d^a^b)+k[3]-722521979|0;c=(c<<16|c>>>16)+d|0;b+=(c^d^a)+k[6]+76029189|0;b=(b<<23|b>>>9)+c|0;a+=(b^c^d)+k[9]-640364487|0;a=(a<<4|a>>>28)+b|0;d+=(a^b^c)+k[12]-421815835|0;d=(d<<11|d>>>21)+a|0;c+=(d^a^b)+k[15]+530742520|0;c=(c<<16|c>>>16)+d|0;b+=(c^d^a)+k[2]-995338651|0;b=(b<<23|b>>>9)+c|0;a+=(c^(b|~d))+k[0]-198630844|0;a=(a<<6|a>>>26)+b|0;d+=(b^(a|~c))+k[7]+1126891415|0;d=(d<<10|d>>>22)+a|0;c+=(a^(d|~b))+k[14]-1416354905|0;c=(c<<15|c>>>17)+d|0;b+=(d^(c|~a))+k[5]-57434055|0;b=(b<<21|b>>>11)+c|0;a+=(c^(b|~d))+k[12]+1700485571|0;a=(a<<6|a>>>26)+b|0;d+=(b^(a|~c))+k[3]-1894986606|0;d=(d<<10|d>>>22)+a|0;c+=(a^(d|~b))+k[10]-1051523|0;c=(c<<15|c>>>17)+d|0;b+=(d^(c|~a))+k[1]-2054922799|0;b=(b<<21|b>>>11)+c|0;a+=(c^(b|~d))+k[8]+1873313359|0;a=(a<<6|a>>>26)+b|0;d+=(b^(a|~c))+k[15]-30611744|0;d=(d<<10|d>>>22)+a|0;c+=(a^(d|~b))+k[6]-1560198380|0;c=(c<<15|c>>>17)+d|0;b+=(d^(c|~a))+k[13]+1309151649|0;b=(b<<21|b>>>11)+c|0;a+=(c^(b|~d))+k[4]-145523070|0;a=(a<<6|a>>>26)+b|0;d+=(b^(a|~c))+k[11]-1120210379|0;d=(d<<10|d>>>22)+a|0;c+=(a^(d|~b))+k[2]+718787259|0;c=(c<<15|c>>>17)+d|0;b+=(d^(c|~a))+k[9]-343485551|0;b=(b<<21|b>>>11)+c|0;x[0]=a+x[0]|0;x[1]=b+x[1]|0;x[2]=c+x[2]|0;x[3]=d+x[3]|0}function md5blk(s){var md5blks=[],i;for(i=0;i<64;i+=4){md5blks[i>>2]=s.charCodeAt(i)+(s.charCodeAt(i+1)<<8)+(s.charCodeAt(i+2)<<16)+(s.charCodeAt(i+3)<<24)}return md5blks}function md5blk_array(a){var md5blks=[],i;for(i=0;i<64;i+=4){md5blks[i>>2]=a[i]+(a[i+1]<<8)+(a[i+2]<<16)+(a[i+3]<<24)}return md5blks}function md51(s){var n=s.length,state=[1732584193,-271733879,-1732584194,271733878],i,length,tail,tmp,lo,hi;for(i=64;i<=n;i+=64){md5cycle(state,md5blk(s.substring(i-64,i)))}s=s.substring(i-64);length=s.length;tail=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];for(i=0;i<length;i+=1){tail[i>>2]|=s.charCodeAt(i)<<(i%4<<3)}tail[i>>2]|=128<<(i%4<<3);if(i>55){md5cycle(state,tail);for(i=0;i<16;i+=1){tail[i]=0}}tmp=n*8;tmp=tmp.toString(16).match(/(.*?)(.{0,8})$/);lo=parseInt(tmp[2],16);hi=parseInt(tmp[1],16)||0;tail[14]=lo;tail[15]=hi;md5cycle(state,tail);return state}function md51_array(a){var n=a.length,state=[1732584193,-271733879,-1732584194,271733878],i,length,tail,tmp,lo,hi;for(i=64;i<=n;i+=64){md5cycle(state,md5blk_array(a.subarray(i-64,i)))}a=i-64<n?a.subarray(i-64):new Uint8Array(0);length=a.length;tail=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];for(i=0;i<length;i+=1){tail[i>>2]|=a[i]<<(i%4<<3)}tail[i>>2]|=128<<(i%4<<3);if(i>55){md5cycle(state,tail);for(i=0;i<16;i+=1){tail[i]=0}}tmp=n*8;tmp=tmp.toString(16).match(/(.*?)(.{0,8})$/);lo=parseInt(tmp[2],16);hi=parseInt(tmp[1],16)||0;tail[14]=lo;tail[15]=hi;md5cycle(state,tail);return state}function rhex(n){var s="",j;for(j=0;j<4;j+=1){s+=hex_chr[n>>j*8+4&15]+hex_chr[n>>j*8&15]}return s}function hex(x){var i;for(i=0;i<x.length;i+=1){x[i]=rhex(x[i])}return x.join("")}if(hex(md51("hello"))!=="5d41402abc4b2a76b9719d911017c592"){add32=function(x,y){var lsw=(x&65535)+(y&65535),msw=(x>>16)+(y>>16)+(lsw>>16);return msw<<16|lsw&65535}}if(typeof ArrayBuffer!=="undefined"&&!ArrayBuffer.prototype.slice){(function(){function clamp(val,length){val=val|0||0;if(val<0){return Math.max(val+length,0)}return Math.min(val,length)}ArrayBuffer.prototype.slice=function(from,to){var length=this.byteLength,begin=clamp(from,length),end=length,num,target,targetArray,sourceArray;if(to!==undefined){end=clamp(to,length)}if(begin>end){return new ArrayBuffer(0)}num=end-begin;target=new ArrayBuffer(num);targetArray=new Uint8Array(target);sourceArray=new Uint8Array(this,begin,num);targetArray.set(sourceArray);return target}})()}function toUtf8(str){if(/[\u0080-\uFFFF]/.test(str)){str=unescape(encodeURIComponent(str))}return str}function utf8Str2ArrayBuffer(str,returnUInt8Array){var length=str.length,buff=new ArrayBuffer(length),arr=new Uint8Array(buff),i;for(i=0;i<length;i+=1){arr[i]=str.charCodeAt(i)}return returnUInt8Array?arr:buff}function arrayBuffer2Utf8Str(buff){return String.fromCharCode.apply(null,new Uint8Array(buff))}function concatenateArrayBuffers(first,second,returnUInt8Array){var result=new Uint8Array(first.byteLength+second.byteLength);result.set(new Uint8Array(first));result.set(new Uint8Array(second),first.byteLength);return returnUInt8Array?result:result.buffer}function hexToBinaryString(hex){var bytes=[],length=hex.length,x;for(x=0;x<length-1;x+=2){bytes.push(parseInt(hex.substr(x,2),16))}return String.fromCharCode.apply(String,bytes)}function SparkMD5(){this.reset()}SparkMD5.prototype.append=function(str){this.appendBinary(toUtf8(str));return this};SparkMD5.prototype.appendBinary=function(contents){this._buff+=contents;this._length+=contents.length;var length=this._buff.length,i;for(i=64;i<=length;i+=64){md5cycle(this._hash,md5blk(this._buff.substring(i-64,i)))}this._buff=this._buff.substring(i-64);return this};SparkMD5.prototype.end=function(raw){var buff=this._buff,length=buff.length,i,tail=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],ret;for(i=0;i<length;i+=1){tail[i>>2]|=buff.charCodeAt(i)<<(i%4<<3)}this._finish(tail,length);ret=hex(this._hash);if(raw){ret=hexToBinaryString(ret)}this.reset();return ret};SparkMD5.prototype.reset=function(){this._buff="";this._length=0;this._hash=[1732584193,-271733879,-1732584194,271733878];return this};SparkMD5.prototype.getState=function(){return{buff:this._buff,length:this._length,hash:this._hash.slice()}};SparkMD5.prototype.setState=function(state){this._buff=state.buff;this._length=state.length;this._hash=state.hash;return this};SparkMD5.prototype.destroy=function(){delete this._hash;delete this._buff;delete this._length};SparkMD5.prototype._finish=function(tail,length){var i=length,tmp,lo,hi;tail[i>>2]|=128<<(i%4<<3);if(i>55){md5cycle(this._hash,tail);for(i=0;i<16;i+=1){tail[i]=0}}tmp=this._length*8;tmp=tmp.toString(16).match(/(.*?)(.{0,8})$/);lo=parseInt(tmp[2],16);hi=parseInt(tmp[1],16)||0;tail[14]=lo;tail[15]=hi;md5cycle(this._hash,tail)};SparkMD5.hash=function(str,raw){return SparkMD5.hashBinary(toUtf8(str),raw)};SparkMD5.hashBinary=function(content,raw){var hash=md51(content),ret=hex(hash);return raw?hexToBinaryString(ret):ret};SparkMD5.ArrayBuffer=function(){this.reset()};SparkMD5.ArrayBuffer.prototype.append=function(arr){var buff=concatenateArrayBuffers(this._buff.buffer,arr,true),length=buff.length,i;this._length+=arr.byteLength;for(i=64;i<=length;i+=64){md5cycle(this._hash,md5blk_array(buff.subarray(i-64,i)))}this._buff=i-64<length?new Uint8Array(buff.buffer.slice(i-64)):new Uint8Array(0);return this};SparkMD5.ArrayBuffer.prototype.end=function(raw){var buff=this._buff,length=buff.length,tail=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],i,ret;for(i=0;i<length;i+=1){tail[i>>2]|=buff[i]<<(i%4<<3)}this._finish(tail,length);ret=hex(this._hash);if(raw){ret=hexToBinaryString(ret)}this.reset();return ret};SparkMD5.ArrayBuffer.prototype.reset=function(){this._buff=new Uint8Array(0);this._length=0;this._hash=[1732584193,-271733879,-1732584194,271733878];return this};SparkMD5.ArrayBuffer.prototype.getState=function(){var state=SparkMD5.prototype.getState.call(this);state.buff=arrayBuffer2Utf8Str(state.buff);return state};SparkMD5.ArrayBuffer.prototype.setState=function(state){state.buff=utf8Str2ArrayBuffer(state.buff,true);return SparkMD5.prototype.setState.call(this,state)};SparkMD5.ArrayBuffer.prototype.destroy=SparkMD5.prototype.destroy;SparkMD5.ArrayBuffer.prototype._finish=SparkMD5.prototype._finish;SparkMD5.ArrayBuffer.hash=function(arr,raw){var hash=md51_array(new Uint8Array(arr)),ret=hex(hash);return raw?hexToBinaryString(ret):ret};return SparkMD5});
